# 機能設計書 57-スパーステンソル操作（Sparse Operations）

## 概要

本ドキュメントは、TensorFlowにおけるスパーステンソル操作機能の設計を記述する。COO（Coordinate）形式のスパーステンソルに対する算術演算、変換、シリアライズ/デシリアライズ、リダクション、結合などのオペレーション群を提供する。

### 本機能の処理概要

**業務上の目的・背景**：自然言語処理のワンホットエンコーディング、推薦システムの疎なユーザ・アイテム行列、グラフニューラルネットワークの隣接行列など、大規模データの多くは疎（スパース）である。スパーステンソルはゼロ要素を省略して格納することでメモリ効率と計算効率を向上させる。本機能はCOO形式（indices, values, dense_shape）で表現されるスパーステンソルの各種操作を提供する。

**機能の利用シーン**：特徴量エンジニアリングでのスパースクロス（SparseCross）、スパースと密テンソルの行列積（SparseTensorDenseMatMul）、スパーステンソルのリダクション（SparseReduceSum/Max）、シリアライズ/デシリアライズによるデータパイプライン内の効率的なスパースデータ転送で利用される。

**主要な処理内容**：
1. 算術演算：SparseAdd, SparseTensorDenseAdd, SparseDenseCwiseMul/Div/Add
2. 行列演算：SparseTensorDenseMatMul
3. 結合・分割：SparseConcat, SparseSplit, SparseCross(V2/Hashed)
4. 変換：SparseToDense, SparseReorder, SparseReshape, SparseSlice
5. リダクション：SparseReduceMax, SparseReduceSum（通常版/Sparse出力版）
6. シリアライズ：SerializeSparse, SerializeManySparse, DeserializeSparse, DeserializeManySparse
7. テンソルマップ：AddSparseToTensorsMap, AddManySparseToTensorsMap, TakeManySparseFromTensorsMap
8. その他：SparseSoftmax, SparseSparseMaximum/Minimum, SparseFillEmptyRows

**関連システム・外部連携**：特になし。

**権限による制御**：特段の権限制御はない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は画面と直接関連しない基盤機能である |

## 機能種別

計算処理 / スパーステンソル操作

## 入力仕様

### 入力パラメータ（代表的なOp）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| indices (a_indices) | int64 | Yes | スパーステンソルのインデックス [nnz, ndims] | rank=2 |
| values (a_values) | T | Yes | スパーステンソルの値 [nnz] | rank=1 |
| shape (a_shape) | int64 | Yes | 密テンソルの形状 [ndims] | rank=1 |
| thresh | Treal | Yes（SparseAdd） | 加算結果の閾値 | rank=0 |
| adjoint_a | bool（属性） | No | 行列Aの随伴行列フラグ（SparseTensorDenseMatMul） | - |
| adjoint_b | bool（属性） | No | 行列Bの随伴行列フラグ（SparseTensorDenseMatMul） | - |

### 入力データソース

計算グラフ内のスパーステンソル表現（indices, values, dense_shape の3タプル）。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| sum_indices/output_indices | int64 | 結果スパーステンソルのインデックス |
| sum_values/output_values | T | 結果スパーステンソルの値 |
| sum_shape/output_shape | int64 | 結果スパーステンソルの形状 |
| product | T | 行列積結果（密テンソル、SparseTensorDenseMatMul） |
| dense | T | 密テンソル変換結果（SparseToDense） |
| serialized_sparse | string/variant | シリアライズ結果 |

### 出力先

後続の計算グラフ。

## 処理フロー

### 処理シーケンス

```
1. スパーステンソル入力の受け取り
   └─ (indices, values, dense_shape) の3タプル検証
2. 各種操作の実行
   ├─ 算術: SparseAdd → 2つのスパーステンソルの加算
   ├─ 行列積: SparseTensorDenseMatMul → スパース x 密の行列積
   ├─ 変換: SparseToDense → 密テンソルへの変換
   ├─ 結合: SparseConcat → 複数スパーステンソルの連結
   └─ リダクション: SparseReduceSum → 指定軸での合計
3. 出力テンソルの返却
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-57-01 | COO形式検証 | indices: rank=2, values: rank=1, shape: rank=1 | 全スパースOp |
| BR-57-02 | インデックス整合性 | indices[0] == values[0]（要素数の一致） | 全スパースOp |
| BR-57-03 | MatMul内部次元一致 | a_shape[adjoint_a ? 0 : 1] == b.shape[adjoint_b ? 1 : 0] | SparseTensorDenseMatMul |
| BR-57-04 | シリアライズ出力 | SerializeSparseは [3] ベクトル、SerializeManySparseは [?, 3] 行列 | シリアライズ系 |
| BR-57-05 | デシリアライズ入力 | serialized_sparseの最後の次元は3 | DeserializeSparse |
| BR-57-06 | テンソルマップステートフル | AddSparseToTensorsMap等はSetIsStatefulが指定 | テンソルマップ系 |

### 計算ロジック

- **SparseTensorDenseMatMul**: スパース行列Aと密行列Bの行列積。adjoint_a/adjoint_bで転置・随伴を制御
- **SparseReduceSum/Max**: 指定軸に沿ったリダクション。keep_dims=trueで次元を保持
- **SparseFillEmptyRows**: 空行をdefault_valueで埋める

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| InvalidArgument | ランクエラー | indices/values/shapeのランクが不正 | 正しい形状で入力 |
| InvalidArgument | 密形状空エラー | dense_shapeが空（SparseFillEmptyRows） | 非空の形状を指定 |

### リトライ仕様

ステートレスな計算であり、リトライに特別な考慮は不要。

## トランザクション仕様

トランザクション制御は行わない。

## パフォーマンス要件

スパース演算の計算量は非ゼロ要素数に比例する。大規模スパースデータでは密テンソルよりも大幅に効率的。

## セキュリティ考慮事項

特段のセキュリティ考慮事項はない。

## 備考

sparse_ops.ccは約660行の大規模ファイルで、約30のオペレーションを定義している。SparseSparseMinOrMaxShapeFn等の共通形状推論関数が複数のOpで再利用されている。SPARSE_DENSE_CWISE_SIGNATUREマクロで3つの要素別演算Opを一括登録している。

---

## コードリーディングガイド

### 推奨読解順序

#### Step 1: データ構造を理解する

COO形式スパーステンソルの3タプル（indices, values, dense_shape）を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | sparse_ops.cc | `tensorflow/core/ops/sparse_ops.cc` | 30-43行目: SparseSparseMinOrMaxShapeFn - indices[rank=2], values[rank=1], shape[rank=1]の検証パターン |

#### Step 2: 主要Opを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | sparse_ops.cc | `tensorflow/core/ops/sparse_ops.cc` | 64-85行目: SparseAdd - 2つのスパーステンソル加算 |
| 2-2 | sparse_ops.cc | `tensorflow/core/ops/sparse_ops.cc` | 87-120行目: SparseTensorDenseMatMul - adjoint_a/adjoint_b付き行列積 |
| 2-3 | sparse_ops.cc | `tensorflow/core/ops/sparse_ops.cc` | 195-209行目: SparseToDense - 密テンソル変換 |
| 2-4 | sparse_ops.cc | `tensorflow/core/ops/sparse_ops.cc` | 483-501行目: SPARSE_DENSE_CWISE_SIGNATUREマクロ - 3つの要素別演算 |
| 2-5 | sparse_ops.cc | `tensorflow/core/ops/sparse_ops.cc` | 602-641行目: SparseFillEmptyRows - 空行埋めの複雑な形状推論 |

### プログラム呼び出し階層図

```
sparse_ops.cc (Op定義)
    |
    +-- ヘルパー関数
    |    +-- SparseSparseMinOrMaxShapeFn (30行目)
    |
    +-- 算術Op
    |    +-- SparseAdd (64行目)
    |    +-- SparseAddGrad (46行目)
    |    +-- SparseTensorDenseAdd (426行目)
    |    +-- SPARSE_DENSE_CWISE_SIGNATURE (483行目)
    |         +-- SparseDenseCwiseMul (497行目)
    |         +-- SparseDenseCwiseDiv (499行目)
    |         +-- SparseDenseCwiseAdd (501行目)
    |
    +-- 行列演算Op
    |    +-- SparseTensorDenseMatMul (87行目)
    |
    +-- 結合・分割Op
    |    +-- SparseConcat (211行目)
    |    +-- SparseCross / V2 / Hashed (256-317行目)
    |    +-- SparseSplit (319行目)
    |    +-- SparseSlice (362行目)
    |
    +-- 変換Op
    |    +-- SparseToDense (195行目)
    |    +-- SparseReorder (385行目)
    |    +-- SparseReshape (406行目)
    |
    +-- リダクションOp
    |    +-- SparseReduceMax / SparseReduceMaxSparse (439-459行目)
    |    +-- SparseReduceSum / SparseReduceSumSparse (461-481行目)
    |
    +-- シリアライズOp
    |    +-- SerializeSparse (122行目)
    |    +-- SerializeManySparse (138行目)
    |    +-- DeserializeSparse (154行目)
    |    +-- DeserializeManySparse (174行目)
    |
    +-- テンソルマップOp
    |    +-- AddSparseToTensorsMap (545行目)
    |    +-- AddManySparseToTensorsMap (563行目)
    |    +-- TakeManySparseFromTensorsMap (581行目)
    |
    +-- その他Op
         +-- SparseSoftmax (505行目)
         +-- SparseSparseMaximum (521行目)
         +-- SparseSparseMinimum (533行目)
         +-- SparseFillEmptyRows (602行目)
         +-- SparseFillEmptyRowsGrad (643行目)
```

### データフロー図

```
[入力]                               [処理]                           [出力]

(indices_a, values_a, shape_a)
+ (indices_b, values_b, shape_b)  -> SparseAdd ------> (sum_indices, sum_values, sum_shape)

(indices, values, shape) + dense  -> SparseTensorDenseMatMul -> product (密テンソル)

(indices, values, shape)          -> SparseToDense ----------> dense (密テンソル)

(indices, values, shape)          -> SerializeSparse --------> serialized [3]
serialized [3]                    -> DeserializeSparse ------> (indices, values, shape)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| sparse_ops.cc | `tensorflow/core/ops/sparse_ops.cc` | ソース | スパーステンソルOp定義（約30 Op） |
| common_shape_fns.h | `tensorflow/core/framework/common_shape_fns.h` | ヘッダ | SparseReduceShapeFn等の共通関数 |
| op.h | `tensorflow/core/framework/op.h` | ヘッダ | REGISTER_OPマクロ定義 |
